#include <iostream>
#include <fstream>
#include <string>
#include <math.h>
#include <algorithm>
#include <tclap/CmdLine.h>


// forward declaration
void text2morse(std::istream& is, std::ofstream& os,
    float wpm = 20, float pitch = 600, float fwpm = 0,
    unsigned long sps = 44100, unsigned short bps = 16);


// subclass CmdLineOutput for custom output
namespace TCLAP {
class CustomOutput : public StdOutput {
    private:
    std::string _progName;

    public:
    CustomOutput() {
        _progName = std::string("text2morse");
    }


    virtual void failure(CmdLineInterface& _cmd, ArgException& e) {
        if ( _cmd.hasHelpAndVersion() ) {
            std::cerr << "usage: " ;
            _shortUsage(_cmd, std::cerr);
        }
    }

    virtual void usage(CmdLineInterface& _cmd ) {
        std::cout << "Usage: ";
        _shortUsage( _cmd, std::cout );
        std::cout << "Options: " << std::endl;
        _longUsage( _cmd, std::cout );
        std::cout << std::endl;
    }

    virtual void version(CmdLineInterface& _cmd) {
        std::string progName = _progName; //_cmd.getProgramName();
        std::string version = _cmd.getVersion();

        std::cout << progName << " " << version << std::endl;
        std::cout << "Copyright (c) 2007 Michael Wichmann, DB6MG" << std::endl;
        std::cout << std::endl;
    }


    void _shortUsage( CmdLineInterface& _cmd, std::ostream& os ) const {
        std::list<Arg*> argList = _cmd.getArgList();
        std::string progName = _progName; //_cmd.getProgramName();
        XorHandler xorHandler = _cmd.getXorHandler();
        std::vector< std::vector<Arg*> > xorList = xorHandler.getXorList();

        std::string s = progName + " ";

        // first the xor
        for ( int i = 0; static_cast<unsigned int>(i) < xorList.size(); i++ )
        {
            s += " {";
            for ( ArgVectorIterator it = xorList[i].begin();
                            it != xorList[i].end(); it++ )
                s += (*it)->shortID() + "|";

            s[s.length()-1] = '}';
        }

        // then the rest
        for (ArgListIterator it = argList.begin(); it != argList.end(); it++)
            if ( !xorHandler.contains( (*it) ) )
                s += " " + (*it)->shortID();

        // if the program name is too long, then adjust the second line offset
        int secondLineOffset = static_cast<int>(progName.length()) + 2;
        if ( secondLineOffset > 75/2 )
                secondLineOffset = static_cast<int>(75/2);

        spacePrint( std::cout, s, 75, 0, secondLineOffset );
    }

    void _longUsage( CmdLineInterface& _cmd, std::ostream& os ) const {
        std::list<Arg*> argList = _cmd.getArgList();
        std::string message = _cmd.getMessage();
        XorHandler xorHandler = _cmd.getXorHandler();
        std::vector< std::vector<Arg*> > xorList = xorHandler.getXorList();

        // first the xor
        for ( int i = 0; static_cast<unsigned int>(i) < xorList.size(); i++ )
        {
            for ( ArgVectorIterator it = xorList[i].begin();
                  it != xorList[i].end();
                  it++ )
            {
                spacePrint( os, (*it)->longID(), 75, 2, 3 );
                spacePrint( os, (*it)->getDescription(), 75, 5, 0 );

                if ( it+1 != xorList[i].end() )
                    spacePrint(os, "-- OR --", 75, 9, 0);
            }
            os << std::endl << std::endl;
        }

        // then the rest
        for (ArgListIterator it = argList.begin(); it != argList.end(); it++)
            if ( !xorHandler.contains( (*it) ) )
            {
                spacePrint( os, (*it)->longID(), 75, 2, 2 );
                spacePrint( os, (*it)->getDescription(), 75, 5, 0 );
                //os << std::endl;
            }

        os << std::endl;

        spacePrint( os, message, 75, 0, 0 );
    }

//    void spacePrint( std::ostream& os,
//						           const std::string& s,
//						           int maxWidth,
//						           int indentSpaces,
//						           int secondLineOffset ) const {


};
} // namespace TCLAP



int main(int argc, char* argv[]) {
    // command line parsing using tclap library
    try {
        // Define the command line object, and insert a message
        // that describes the program. The "Command description message"
        // is printed last in the help text. The second argument is the
        // delimiter (usually space) and the last one is the version number.
        // The CmdLine object parses the argv array based on the Arg objects
        // that it contains.
        TCLAP::CmdLine cmd("Converts text files to morse sound files", ' ', "0.2");
        TCLAP::CustomOutput outp;
        cmd.setOutput(&outp);

        // Define a value argument and add it to the command line.
        // A value arg defines a flag and a type of value that it expects,
        // such as "-n Bishop".
        //TCLAP::ValueArg<std::string> nameArg("n","name","Name to print",true,"homer","string");

        // Add the argument nameArg to the CmdLine object. The CmdLine object
        // uses this Arg to parse the command line.
        //cmd.add( nameArg );

        // Define a switch and add it to the command line.
        // A switch arg is a boolean argument and only defines a flag that
        // indicates true or false.  In this example the SwitchArg adds itself
        // to the CmdLine object as part of the constructor.  This eliminates
        // the need to call the cmd.add() method.  All args have support in
        // their constructors to add themselves directly to the CmdLine object.
        // It doesn't matter which idiom you choose, they accomplish the same thing.
        //TCLAP::SwitchArg reverseSwitch("r","reverse","Print name backwards", cmd, false);


        // Define command line arguments
        TCLAP::ValueArg<std::string> outputFilenameArg("o", "outfile",
            "Output filename", true, "cwout.wav", "outfile");

        TCLAP::ValueArg<std::string> inputFilenameArg("i", "infile",
            "Input text file. If none is given read from stdin", false, "", "infile");

        TCLAP::ValueArg<float> wpmArg("w", "wpm",
            "Speed in words per minute (PARIS), default=20wpm", false, 20, "wpm");

        TCLAP::ValueArg<float> pitchArg("p", "pitch",
            "Tone pitch in Hz, default=600Hz", false, 600, "pitch");

        TCLAP::ValueArg<float> fwpmArg("f", "farnsworthwpm",
            "Farnsworth wpm", false, 0, "wpm");

        //TCLAP::ValueArg<int> qrnArg("n", "qrn", "QRN", false, 0, "0-5");

        //cmd.add(qrnArg);
        cmd.add(fwpmArg);
        cmd.add(pitchArg);
        cmd.add(wpmArg);
        cmd.add(inputFilenameArg);
        cmd.add(outputFilenameArg);

        // Parse the argv array.
        cmd.parse( argc, argv );

        // Get the value parsed by each arg.
        std::string outputFilename = outputFilenameArg.getValue();
        std::string inputFilename = inputFilenameArg.getValue();

        float pitch = pitchArg.getValue();
        float wpm = wpmArg.getValue();
        float fwpm = fwpmArg.getValue();

        // do what you wanted
        std::istream *instream;
        std::ifstream infilestr;
        std::ofstream outfilestr;


        // print out version
        outp.version(cmd);

        // open output file
        outfilestr.open(outputFilename.c_str(), std::ios::out|std::ios::binary);
        if (!outfilestr.good()) {
            std::cout << "error opening file: " << outputFilename << std::endl;
            exit(1);
        }
        else {
            std::cout << "opening file for output: " << outputFilename << std::endl;
        }

        // open input file
        if (!inputFilename.empty()) {
            infilestr.open(inputFilename.c_str(), std::ios::in);
            if (infilestr.good()) {
                std::cout << "opening file for input: " << inputFilename << std::endl;
                instream = &infilestr;
            }
            else {
                std::cout << "error opening file: " << inputFilename << std::endl;
                exit(1);
            }
        }
        else {
            std::cout << "reading input from stdin" << std::endl;
            instream = &std::cin;
        }

        text2morse(*instream, outfilestr, wpm, pitch, fwpm, 22050);

        outfilestr.close();
        infilestr.close();

    }
    // catch any exceptions
    catch (TCLAP::ArgException &e) {
        std::cerr << "error: " << e.error() << " for arg " << e.argId() << std::endl;
    }

    return 0;
}

//using namespace std;


typedef struct {
    char    id[4];
    long    chunkSize;
    char    type[4];
} RiffChunk;

typedef struct {
    char            id[4];
    long            chunkSize;
    short           wFormatTag;
    unsigned short  wChannels;
    unsigned long   dwSamplesPerSec;
    unsigned long   dwAvgBytesPerSec;
    unsigned short  wBlockAlign;
    unsigned short  wBitsPerSample;
} FormatChunk;


typedef struct {
    char            id[4];
    long            chunkSize;
    unsigned char   waveformData[];
} DataChunk;


void text2morse(std::istream& is, std::ofstream& os,
    float wpm, float pitch, float fwpm, unsigned long sps, unsigned short bps) {
    char ch;
    long filesize;

    // write wav header
    RiffChunk riffChunk;
    FormatChunk formatChunk;
    DataChunk dataChunk;

    std::string("RIFF").copy(riffChunk.id, 4);
    riffChunk.chunkSize = 0;
    std::string("WAVE").copy(riffChunk.type, 4);
    os.write((char*) &riffChunk, sizeof(riffChunk));

    std::string("fmt ").copy(formatChunk.id, 4);
    formatChunk.chunkSize = 16;
    formatChunk.wFormatTag = 1; // 1 = uncompressed
    formatChunk.wChannels = 1;
    // samples per second
    formatChunk.dwSamplesPerSec = sps;
    // wblockalign = wchannels * wbistspersample / 8
    formatChunk.wBlockAlign = (unsigned short) ceil(1. * (float)bps/8.);
    // avgbytespersec = dwsamplespersec * wblockalign
    formatChunk.dwAvgBytesPerSec = formatChunk.dwSamplesPerSec * formatChunk.wBlockAlign;
    formatChunk.wBitsPerSample = bps;
    os.write((char*) &formatChunk, sizeof(formatChunk));

    std::string("data").copy(dataChunk.id, 4);
    dataChunk.chunkSize = 0;
    os.write((char*) &dataChunk, sizeof(dataChunk)-sizeof(unsigned char*));


    // go through input stream and convert to morse

    float dottime = 6./(5.*wpm);
    float dt = 1./(float)sps;
    float time = 0;
    float stretch = 1;
    short sh;
    std::string ddq;
    std::string::iterator ddq_iter;

    if ((fwpm < wpm) and (fwpm > 0)) {
        stretch = (25.*wpm/fwpm - 19.)/6;
    }

    while (!is.eof()) {
        ch = is.get();

        switch (ch) {
            case '0':
                ddq = std::string("-----");
                break;
            case '1':
                ddq = std::string(".----");
                break;
            case '2':
                ddq = std::string("..---");
                break;
            case '3':
                ddq = std::string("...--");
                break;
            case '4':
                ddq = std::string("....-");
                break;
            case '5':
                ddq = std::string(".....");
                break;
            case '6':
                ddq = std::string("-....");
                break;
            case '7':
                ddq = std::string("--...");
                break;
            case '8':
                ddq = std::string("---..");
                break;
            case '9':
                ddq = std::string("----.");
                break;
            case 'a':
                ddq = std::string(".-");
                break;
            case 'b':
                ddq = std::string("-...");
                break;
            case 'c':
                ddq = std::string("-.-.");
                break;
            case 'd':
                ddq = std::string("-..");
                break;
            case 'e':
                ddq = std::string(".");
                break;
            case 'f':
                ddq = std::string("..-.");
                break;
            case 'g':
                ddq = std::string("--.");
                break;
            case 'h':
                ddq = std::string("....");
                break;
            case 'i':
                ddq = std::string("..");
                break;
            case 'j':
                ddq = std::string(".---");
                break;
            case 'k':
                ddq = std::string("-.-");
                break;
            case 'l':
                ddq = std::string(".-..");
                break;
            case 'm':
                ddq = std::string("--");
                break;
            case 'n':
                ddq = std::string("-.");
                break;
            case 'o':
                ddq = std::string("---");
                break;
            case 'p':
                ddq = std::string(".--.");
                break;
            case 'q':
                ddq = std::string("--.-");
                break;
            case 'r':
                ddq = std::string(".-.");
                break;
            case 's':
                ddq = std::string("...");
                break;
            case 't':
                ddq = std::string("-");
                break;
            case 'u':
                ddq = std::string("..-");
                break;
            case 'v':
                ddq = std::string("...-");
                break;
            case 'w':
                ddq = std::string(".--");
                break;
            case 'x':
                ddq = std::string("-..-");
                break;
            case 'y':
                ddq = std::string("-.--");
                break;
            case 'z':
                ddq = std::string("--..");
                break;
            case '+':
                ddq = std::string(".-.-.");
                break;
            case '=':
                ddq = std::string("-...-");
                break;
            case '/':
                ddq = std::string("-..-.");
                break;
            case ' ':
            case '\n':
                ddq = std::string(" ");
                break;
            default:
                ddq = std::string("xx");
        }

        for(ddq_iter = ddq.begin(); ddq_iter != ddq.end(); ddq_iter++) {
            // non printable character break out of for loop
            if (*ddq_iter == 'x') break;

            // sound a dash
            if (*ddq_iter == '-') {
                for (time = 0; time <= 3.*dottime; time += dt) {
                    sh = (short) (32510. * sin(6.28*pitch*time) ) ;
                    os.write((char*) &sh, 2);
                }
            }
            // sound a dot
            else if (*ddq_iter == '.') {
                for (time = 0; time <= 1.*dottime; time += dt) {
                    sh = (short) (32510. * sin(6.28*pitch*time) ) ;
                    os.write((char*) &sh, 2);
                }
            }
            // sound a word space (6 dots)
            else if (*ddq_iter == ' ') {
                for (time = 0; time <= 6.*dottime*stretch; time += dt) {
                    sh = (short) (0) ;
                    os.write((char*) &sh, 2);
                }
            }



            if ((ddq_iter != ddq.end()) and (*ddq_iter != ' ')) {
                // plus an additional dot as symbol space
                for (time = 0; time <= 1.*dottime; time += dt) {
                        sh = (short) (0) ;
                        os.write((char*) &sh, 2);
                }
            }
        }

        if (ddq_iter == ddq.end()) {
            // after every ddq entry leave letter space (3 dots)
            for (time = 0; time <= 3.*dottime*stretch; time += dt) {
                sh = (short) (0)  ;
                os.write((char*) &sh, 2);
            }
        }

    }

    // set sizes

    filesize = (long) os.tellp();

    os.seekp(4, std::ios::beg);
    riffChunk.chunkSize = filesize - 8;
    os.write((char*) &riffChunk.chunkSize, 4);

    os.seekp(42, std::ios::beg);
    dataChunk.chunkSize = filesize - 42;
    os.write((char*) &dataChunk.chunkSize, 4);


}

